<?php
if ( class_exists( 'JMT_Settings' ) ) return;

class JMT_Settings {

    private static $instance;
    private $option_name = 'jmt_settings';

    private function is_nl_admin() {
        $loc = function_exists( 'get_user_locale' ) ? get_user_locale() : ( function_exists( 'determine_locale' ) ? determine_locale() : get_locale() );
        $loc = strtolower( (string) $loc );
        return substr( $loc, 0, 2 ) === 'nl';
    }

    private function t( $nl, $en ) {
        return $this->is_nl_admin() ? $nl : $en;
    }

    public static function instance() {
        return self::$instance ?: self::$instance = new self();
    }

    private function __construct() {
        add_action( 'admin_init', [ $this, 'register_settings' ] );
        add_action( 'wp_ajax_jmt_clear_cache',    [ $this, 'ajax_clear_cache' ] );
        add_action( 'wp_ajax_jmt_get_cache_info', [ $this, 'ajax_get_cache_info' ] );
        add_action( 'wp_ajax_jmt_force_refresh',  [ $this, 'ajax_force_refresh' ] );
    }

    public function register_settings() {
        register_setting( 'jmt_settings_group', $this->option_name, [
            'sanitize_callback' => [ $this, 'sanitize_settings' ],
        ] );

        add_settings_section( 'jmt_main', $this->t( 'Algemene instellingen', 'General settings' ), null, 'jmt-settings' );

        add_settings_field( 'items_per_page', $this->t( 'Producten per pagina', 'Products per page' ),
            [ $this, 'field_items_per_page' ], 'jmt-settings', 'jmt_main' );
        add_settings_field( 'notification_email', $this->t( 'Notificatie-e-mail', 'Notification email' ),
            [ $this, 'field_notification_email' ], 'jmt-settings', 'jmt_main' );
        add_settings_field( 'cache_expiration', $this->t( 'Cache vervaltijd (uren)', 'Cache expiration (hours)' ),
            [ $this, 'field_cache_expiration' ], 'jmt-settings', 'jmt_main' );
    }

    public function sanitize_settings( $input ) {
        return [
            'items_per_page'    => max( 1, absint( $input['items_per_page'] ?? 20 ) ),
            'notification_email'=> sanitize_email( $input['notification_email'] ?? '' ),
            'cache_expiration'  => max( 1, absint( $input['cache_expiration'] ?? 12 ) ),
        ];
    }

    public function field_items_per_page() {
        $v = $this->get( 'items_per_page', 20 );
        echo '<input type="number" name="' . esc_attr( $this->option_name ) . '[items_per_page]" value="' . esc_attr( $v ) . '" min="1" class="small-text">';
    }

    public function field_notification_email() {
        $v = $this->get( 'notification_email', '' );
        echo '<input type="email" name="' . esc_attr( $this->option_name ) . '[notification_email]" value="' . esc_attr( $v ) . '" class="regular-text">';
    }

    public function field_cache_expiration() {
        $v = $this->get( 'cache_expiration', 12 );
        echo '<input type="number" name="' . esc_attr( $this->option_name ) . '[cache_expiration]" value="' . esc_attr( $v ) . '" min="1" class="small-text"> ' . esc_html( $this->t( 'uren', 'hours' ) );
    }

    // ── AJAX: clear cache ────────────────────────────────────────────────────

    public function ajax_clear_cache() {
        check_ajax_referer( 'jmt_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => $this->t( 'Geen toegang.', 'Access denied.' ) ] );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'jmt_products';
        $wpdb->query( "TRUNCATE TABLE {$table}" );

        $files = glob( JMTB_DIR . 'cache/jmt_products_*.json' );
        if ( is_array( $files ) ) {
            foreach ( $files as $f ) {
                @unlink( $f );
            }
        }

        // Delete fetch-lock transients so an immediate refresh is possible
        delete_transient( 'jmt_fetching_nl_NL' );
        delete_transient( 'jmt_fetching_en_GB' );

        wp_send_json_success( [
            'message'    => $this->t( 'Cache succesvol verwijderd.', 'Cache cleared successfully.' ),
            'cache_html' => $this->render_cache_info_html(),
        ] );
    }

    // ── AJAX: force refresh (clear + trigger background fetch) ───────────────

    public function ajax_force_refresh() {
        check_ajax_referer( 'jmt_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => $this->t( 'Geen toegang.', 'Access denied.' ) ] );
        }

        global $wpdb;
        $table = $wpdb->prefix . 'jmt_products';
        $wpdb->query( "TRUNCATE TABLE {$table}" );

        $files = glob( JMTB_DIR . 'cache/jmt_products_*.json' );
        if ( is_array( $files ) ) {
            foreach ( $files as $f ) {
                @unlink( $f );
            }
        }

        delete_transient( 'jmt_fetching_nl_NL' );
        delete_transient( 'jmt_fetching_en_GB' );
        JMT_Fetcher::instance()->trigger_background_fetch( 'nl_NL' );
        JMT_Fetcher::instance()->trigger_background_fetch( 'en_GB' );

        wp_send_json_success( [
            'message'    => $this->t( 'Cache verwijderd. Productlijst wordt opnieuw opgehaald van de leverancier…', 'Cache cleared. Product list is being fetched from the supplier…' ),
            'cache_html' => $this->render_cache_info_html(),
        ] );
    }

    // ── AJAX: get cache info (for live refresh of the panel) ─────────────────

    public function ajax_get_cache_info() {
        check_ajax_referer( 'jmt_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( [ 'message' => $this->t( 'Geen toegang.', 'Access denied.' ) ] );
        }

        wp_send_json_success( [ 'cache_html' => $this->render_cache_info_html() ] );
    }

    // ── Cache info HTML fragment ──────────────────────────────────────────────

    /**
     * Returns the HTML for the cache-status panel.
     * Called both on page load (PHP) and via AJAX (after clear/refresh).
     */
    public function render_cache_info_html() {
        $info = JMT_Fetcher::instance()->get_cache_info();

        ob_start();

        if ( $info['fetching'] ) {
            echo '<p class="jmt-ci-row jmt-ci-fetching">'
               . '⏳ <strong>' . esc_html( $this->t( 'Bezig met ophalen van productlijst…', 'Fetching product list…' ) ) . '</strong> ' . esc_html( $this->t( 'even geduld.', 'please wait.' ) )
               . '</p>';
        }

        if ( $info['has_data'] ) {
            $age_s  = (int) $info['age_seconds'];
            $hours  = (int) floor( $age_s / 3600 );
            $mins   = (int) floor( ( $age_s % 3600 ) / 60 );

            if ( $hours > 0 ) {
                $age_str = $this->is_nl_admin()
                    ? ( $hours . ' uur en ' . $mins . ' minuten' )
                    : ( $hours . ' hours and ' . $mins . ' minutes' );
            } elseif ( $mins > 0 ) {
                $age_str = $this->is_nl_admin() ? ( $mins . ' minuten' ) : ( $mins . ' minutes' );
            } else {
                $age_str = $this->t( 'minder dan een minuut', 'less than a minute' );
            }

            $bytes = (int) $info['size_bytes'];
            if ( $bytes >= 1048576 ) {
                $size_str = number_format( $bytes / 1048576, 1, ',', '.' ) . ' MB';
            } elseif ( $bytes >= 1024 ) {
                $size_str = number_format( $bytes / 1024, 0, ',', '.' ) . ' KB';
            } else {
                $size_str = $bytes . ' bytes';
            }

            echo '<table class="jmt-ci-table">';
            echo '<tr><th>' . esc_html( $this->t( 'Laatste update', 'Last update' ) ) . '</th><td>' . esc_html( $age_str ) . ' ' . esc_html( $this->t( 'geleden', 'ago' ) ) . '</td></tr>';
            echo '<tr><th>' . esc_html( $this->t( 'Tijdstip', 'Timestamp' ) ) . '</th><td>'       . esc_html( $info['created_at'] ?? '–' ) . '</td></tr>';
            echo '<tr><th>' . esc_html( $this->t( 'Cache grootte', 'Cache size' ) ) . '</th><td>'  . esc_html( $size_str ) . '</td></tr>';
            echo '<tr><th>' . esc_html( $this->t( 'Opgeslagen in', 'Stored in' ) ) . '</th><td>'  . esc_html( ucfirst( $info['source'] ?? '–' ) ) . '</td></tr>';
            echo '</table>';
        } else {
            if ( ! $info['fetching'] ) {
                echo '<p class="jmt-ci-row jmt-ci-empty">❌ ' . esc_html( $this->t( 'Geen cache beschikbaar.', 'No cache available.' ) ) . '</p>';
            }
        }

        return ob_get_clean();
    }

    // ── Render settings page ─────────────────────────────────────────────────

    public function render_settings_page() {
        $nonce = wp_create_nonce( 'jmt_nonce' );
        ?>
        <form method="post" action="options.php">
            <?php
            settings_fields( 'jmt_settings_group' );
            do_settings_sections( 'jmt-settings' );
            submit_button();
            ?>
        </form>

        <h2><?php echo esc_html( $this->t( 'Cache beheer', 'Cache management' ) ); ?></h2>

        <div id="jmt-cache-info-panel">
            <?php echo $this->render_cache_info_html(); ?>
        </div>

        <p style="margin-top:0.75em;">
            <button id="jmt-clear-cache"   class="button"><?php echo esc_html( $this->t( 'Cache verwijderen', 'Clear cache' ) ); ?></button>
            <button id="jmt-force-refresh" class="button button-primary" style="margin-left:0.5em;"><?php echo esc_html( $this->t( 'Nu vernieuwen', 'Refresh now' ) ); ?></button>
            <button id="jmt-refresh-info"  class="button" style="margin-left:0.5em;">↻ <?php echo esc_html( $this->t( 'Status vernieuwen', 'Refresh status' ) ); ?></button>
            <span id="jmt-cache-msg" style="margin-left:1em;"></span>
        </p>

        <script>
        jQuery(function($){
            var nonce = '<?php echo esc_js( $nonce ); ?>';

            function showMsg(msg, ok) {
                var color = ok ? 'green' : '#c00';
                $('#jmt-cache-msg').html('<span style="color:' + color + '">' + msg + '</span>');
                setTimeout(function(){ $('#jmt-cache-msg').empty(); }, 4000);
            }

            function updatePanel(html) {
                $('#jmt-cache-info-panel').html(html);
            }

            var I18N = <?php echo wp_json_encode([
                'confirm_clear' => $this->t( 'Cache verwijderen?', 'Clear cache?' ),
                'busy'          => $this->t( 'Bezig…', 'Working…' ),
                'refresh_now'   => $this->t( 'Nu vernieuwen', 'Refresh now' ),
                'error'         => $this->t( 'Fout.', 'Error.' ),
            ]); ?>;

            // ── Clear cache ──────────────────────────────────────────────────
            $('#jmt-clear-cache').on('click', function(){
                if (!confirm(I18N.confirm_clear)) return;
                $.post(ajaxurl, { action:'jmt_clear_cache', nonce:nonce }, function(r){
                    if (r.success) {
                        showMsg(r.data.message, true);
                        updatePanel(r.data.cache_html);
                    } else {
                        showMsg((r.data && r.data.message) || I18N.error, false);
                    }
                });
            });

            // ── Force refresh ────────────────────────────────────────────────
            $('#jmt-force-refresh').on('click', function(){
                $(this).prop('disabled', true).text(I18N.busy);
                var btn = this;
                $.post(ajaxurl, { action:'jmt_force_refresh', nonce:nonce }, function(r){
                    $(btn).prop('disabled', false).text(I18N.refresh_now);
                    if (r.success) {
                        showMsg(r.data.message, true);
                        updatePanel(r.data.cache_html);
                        // Poll status every 5 s while fetching
                        var polls = 0;
                        var interval = setInterval(function(){
                            polls++;
                            $.post(ajaxurl, { action:'jmt_get_cache_info', nonce:nonce }, function(r2){
                                if (r2.success) updatePanel(r2.data.cache_html);
                                // Stop polling after 12 attempts (≈ 1 min) or when not fetching
                                if (polls >= 12 || $('#jmt-cache-info-panel .jmt-ci-fetching').length === 0) {
                                    clearInterval(interval);
                                }
                            });
                        }, 5000);
                    } else {
                        showMsg((r.data && r.data.message) || I18N.error, false);
                    }
                });
            });

            // ── Refresh status display ───────────────────────────────────────
            $('#jmt-refresh-info').on('click', function(){
                $.post(ajaxurl, { action:'jmt_get_cache_info', nonce:nonce }, function(r){
                    if (r.success) updatePanel(r.data.cache_html);
                });
            });
        });
        </script>
        <?php
    }

    // ── Getters ───────────────────────────────────────────────────────────────

    private function get( $key, $default = '' ) {
        $opts = get_option( $this->option_name, [] );
        return $opts[ $key ] ?? $default;
    }

    public function get_feed_url( $lang = null ) {
        // Default behavior: follow site locale (for backwards compatibility).
        if ( $lang === null ) {
            $locale = function_exists( 'determine_locale' ) ? determine_locale() : get_locale();
            $locale = strtolower( (string) $locale );
            $lang = ( substr( $locale, 0, 2 ) === 'nl' ) ? 'nl_NL' : 'en_GB';
        }

        $lang = str_replace( '-', '_', (string) $lang );
        $lang_lc = strtolower( $lang );
        if ( str_starts_with( $lang_lc, 'nl' ) ) $lang = 'nl_NL';
        else $lang = 'en_GB';

        return 'https://www.jmt.nl/' . $lang . '/products.json';
    }

    public function get_items_per_page() {
        return (int) $this->get( 'items_per_page', 20 );
    }

    public function get_notification_email() {
        $email = $this->get( 'notification_email', '' );
        return $email ?: get_option( 'admin_email' );
    }

    /** Returns cache expiration in seconds */
    public function get_cache_expiration() {
        return (int) $this->get( 'cache_expiration', 12 ) * HOUR_IN_SECONDS;
    }
}
