<?php
if ( class_exists( 'JMT_Frontend' ) ) return;

class JMT_Frontend {

    private static $instance;

    /** Maps JMT taxon codes to Dutch display names */
    private $category_map = [
        'ST'  => 'Stoelen',
        'VK'  => 'Vergader- & Kantoormeubilair',
        'KD'  => 'Keukeninrichting & Divers',
        'VE'  => 'Verkoop',
        'FS'  => 'Fauteuils, Sofa\'s & Lounge',
        'FO'  => 'Folder- & Opbergrekken',
        'DS'  => 'Toebehoren',
        'BK'  => 'Barkrukken & Krukken',
        'VD'  => 'Vitrines, Displays & Balies',
        'TT'  => 'Tafels & Tafelbladen',
        'TAP' => 'Tapijt',
        ''    => 'Overig',
    ];

    /** Maps JMT taxon codes to English display names */
    private $category_map_en = [
        'ST'  => 'Chairs',
        'VK'  => 'Meeting & Office furniture',
        'KD'  => 'Kitchen & Misc',
        'VE'  => 'Sales',
        'FS'  => 'Armchairs, Sofas & Lounge',
        'FO'  => 'Brochure & Storage racks',
        'DS'  => 'Accessories',
        'BK'  => 'Bar stools & Stools',
        'VD'  => 'Showcases, Displays & Counters',
        'TT'  => 'Tables & Table tops',
        'TAP' => 'Carpet',
        ''    => 'Other',
    ];

    public static function instance() {
        return self::$instance ?: self::$instance = new self();
    }

    private function __construct() {
        add_shortcode( 'jmt_product_browser', [ $this, 'render_browser' ] );
        add_action( 'wp_enqueue_scripts',           [ $this, 'enqueue_assets' ] );
        add_action( 'wp_ajax_jmt_get_products',     [ $this, 'ajax_get_products' ] );
        add_action( 'wp_ajax_nopriv_jmt_get_products', [ $this, 'ajax_get_products' ] );
        add_action( 'wp_ajax_jmt_send_quote',       [ $this, 'ajax_send_quote' ] );
        add_action( 'wp_ajax_nopriv_jmt_send_quote', [ $this, 'ajax_send_quote' ] );
    }

    // ── Assets ────────────────────────────────────────────────────────────────

    public function enqueue_assets() {
        global $post;
        if ( ! $post || ! has_shortcode( $post->post_content, 'jmt_product_browser' ) ) return;

        wp_enqueue_style(
            'jmt-style',
            JMTB_URL . 'assets/css/style.css',
            [],
            JMTB_VERSION
        );
        wp_enqueue_script(
            'jmt-frontend',
            JMTB_URL . 'assets/js/frontend.js',
            [ 'jquery' ],
            JMTB_VERSION,
            true
        );
        wp_localize_script( 'jmt-frontend', 'JMT_Ajax', [
            'ajax_url' => admin_url( 'admin-ajax.php' ),
            'nonce'    => wp_create_nonce( 'jmt_nonce' ),
            'locale'   => function_exists( 'determine_locale' ) ? determine_locale() : get_locale(),
        ] );
    }

    // ── Shortcode HTML ────────────────────────────────────────────────────────

    public function render_browser( $atts ) {
        $is_nl = $this->is_dutch_locale();
        ob_start();
        ?>
        <button type="button" class="jmt-menu-toggle" aria-expanded="false">
            <span class="jmt-menu-toggle-icon">&#9776;</span>
            <span><?php echo esc_html( $is_nl ? 'Categorieën & Kleuren' : 'Categories & Colours' ); ?></span>
        </button>

        <div class="jmt-container">

            <aside class="jmt-sidebar">

                <div class="jmt-accordion jmt-accordion--cats" data-acc="cats">
                    <button type="button" class="jmt-acc-toggle" aria-expanded="true">
                        <span class="jmt-acc-title"><?php echo esc_html( $is_nl ? 'Categorieën' : 'Categories' ); ?></span>
                        <span class="jmt-acc-icon" aria-hidden="true">▾</span>
                    </button>
                    <div class="jmt-acc-panel" data-panel="cats">
                        <ul id="jmt-category-filter" class="jmt-category-list"></ul>
                    </div>
                </div>

                <div id="jmt-color-section" class="jmt-accordion jmt-accordion--colors is-collapsed" data-acc="colors" style="display:none;">
                    <button type="button" class="jmt-acc-toggle" aria-expanded="false">
                        <span class="jmt-acc-title"><?php echo esc_html( $is_nl ? 'Kleuren' : 'Colours' ); ?></span>
                        <span class="jmt-acc-icon" aria-hidden="true">▾</span>
                    </button>
                    <div class="jmt-acc-panel" data-panel="colors">
                        <ul id="jmt-color-filter" class="jmt-color-list"></ul>
                    </div>
                </div>

                <button id="jmt-sidebar-quote" class="jmt-button"><?php echo esc_html( $is_nl ? 'Offerte aanvragen' : 'Request a quote' ); ?></button>
                <p class="jmt-version">JMT Browser v<?php echo esc_html( JMTB_VERSION ); ?></p>
            </aside>

            <main class="jmt-main">

                <div id="jmt-products" class="jmt-products-grid">
                    <p class="jmt-loading"><?php echo esc_html( $is_nl ? 'Producten worden geladen…' : 'Loading products…' ); ?></p>
                </div>

                <!-- Product detail modal -->
                <div id="jmt-modal" class="jmt-modal" style="display:none;">
                    <div class="jmt-modal-content">
                        <span id="jmt-modal-close" class="jmt-modal-close">&times;</span>
                        <div id="jmt-modal-body"></div>
                    </div>
                </div>

                <!-- Quote request form -->
                <form id="jmt-quote-form" class="jmt-form" method="post">
                    <?php wp_nonce_field( 'jmt_nonce', 'nonce' ); ?>
                    <input type="hidden" name="action" value="jmt_send_quote">
                    <input type="hidden" name="cart"   id="jmt-cart-input" value="">
                    <h3><?php echo esc_html( $is_nl ? 'Offerte aanvragen' : 'Request a quote' ); ?></h3>
                    <div class="jmt-form-row"><label><?php echo esc_html( $is_nl ? 'Naam *' : 'Name *' ); ?></label>    <input type="text"  name="name"       required></div>
                    <div class="jmt-form-row"><label><?php echo esc_html( $is_nl ? 'E-mail *' : 'Email *' ); ?></label>   <input type="email" name="email"      required></div>
                    <div class="jmt-form-row"><label><?php echo esc_html( $is_nl ? 'Bedrijf' : 'Company' ); ?></label>    <input type="text"  name="company"></div>
                    <div class="jmt-form-row"><label><?php echo esc_html( $is_nl ? 'Event' : 'Event' ); ?></label>      <input type="text"  name="event_name"></div>
                    <div class="jmt-form-row"><label><?php echo esc_html( $is_nl ? 'Datum' : 'Date' ); ?></label>      <input type="date"  name="event_date"></div>
                    <ul id="jmt-cart" class="jmt-cart"></ul>
                    <p id="jmt-cart-empty" class="jmt-cart-empty"><?php echo esc_html( $is_nl ? 'Uw winkelwagen is leeg.' : 'Your cart is empty.' ); ?></p>
                    <button type="submit" class="jmt-button" id="jmt-submit-quote"><?php echo esc_html( $is_nl ? 'Verstuur aanvraag' : 'Send request' ); ?></button>
                </form>

            </main>
        </div>
        <?php
        return ob_get_clean();
    }

    // ── AJAX: get products ────────────────────────────────────────────────────

    public function ajax_get_products() {
        check_ajax_referer( 'jmt_nonce', 'nonce' );

        $lang_raw = sanitize_text_field( wp_unslash( $_POST['lang'] ?? '' ) );
        $lang_lc  = strtolower( str_replace( '-', '_', $lang_raw ) );
        $lang     = ( $lang_lc !== '' && str_starts_with( $lang_lc, 'nl' ) ) ? 'nl_NL' : 'en_GB';
        $is_nl    = ( $lang === 'nl_NL' );

        $page     = max( 1, (int) ( $_POST['page']     ?? 1 ) );
        $category = sanitize_text_field( wp_unslash( $_POST['category'] ?? '' ) );
        $color    = sanitize_text_field( wp_unslash( $_POST['color']    ?? '' ) );
        $per_page = JMT_Settings::instance()->get_items_per_page();

        $all = JMT_Fetcher::instance()->get_products( $lang );

        if ( empty( $all ) ) {
            wp_send_json_error( [
                'message' => $is_nl
                    ? 'Productlijst wordt opgehaald van de leverancier, even geduld…'
                    : 'Product list is being fetched from the supplier, please wait…',
                'status'  => 'warming',
            ] );
            return;
        }

        // Collect categories and colors from the full unfiltered list
        $all_categories = $this->collect_categories( $all, $is_nl );
        $all_colors     = $this->collect_colors( $all );

        // Filter by selected category (used for paging) — also used to derive
        // a category-specific colour list for the sidebar.
        $filtered_by_cat = $category
            ? array_values( array_filter( $all, function( $item ) use ( $category, $is_nl ) {
                return $this->taxon_to_label( $item['taxons']['main'] ?? '', $is_nl ) === $category;
            } ) )
            : array_values( $all );

        $colors_in_category = $category
            ? $this->collect_colors( $filtered_by_cat )
            : $all_colors;

        $filtered = $filtered_by_cat;

        // Filter by selected primary colour
        if ( $color ) {
            $filtered = array_values( array_filter( $filtered, function( $item ) use ( $color ) {
                return $this->color_slug( $this->get_primary_color( $item['attributes'] ?? [] ) ) === $color;
            } ) );
        }

        $total  = count( $filtered );
        $offset = ( $page - 1 ) * $per_page;
        $paged  = array_slice( $filtered, $offset, $per_page );

        $products = array_map( function( $it ) use ( $is_nl ) { return $this->format_product( $it, $is_nl ); }, $paged );

        // Build a slim variant lookup keyed by base_name so we can attach
        // colour-variant swatches to each product popup.
        $variant_lookup = $this->build_variant_lookup( $all );

        foreach ( $products as &$prod ) {
            $base = $prod['base_name'] ?? '';
            if ( $base && isset( $variant_lookup[ $base ] ) ) {
                // Exclude self
                $prod['variants'] = array_values( array_filter(
                    $variant_lookup[ $base ],
                    function( $v ) use ( $prod ) { return $v['id'] !== $prod['id']; }
                ) );
            } else {
                $prod['variants'] = [];
            }
        }
        unset( $prod );

        wp_send_json_success( [
            'products'   => $products,
            'total'      => $total,
            'per_page'   => $per_page,
            'page'       => $page,
            'categories' => $all_categories,
            'colors'     => $all_colors,
            'colors_in_category' => $colors_in_category,
        ] );
    }

    // ── AJAX: send quote ──────────────────────────────────────────────────────

    public function ajax_send_quote() {
        if ( ! check_ajax_referer( 'jmt_nonce', 'nonce', false ) ) {
            wp_send_json_error( [ 'message' => 'Beveiligingsfout: ongeldige nonce.' ] );
            return;
        }

        $data = [
            'name'       => sanitize_text_field( wp_unslash( $_POST['name']       ?? '' ) ),
            'email'      => sanitize_email(       wp_unslash( $_POST['email']      ?? '' ) ),
            'company'    => sanitize_text_field( wp_unslash( $_POST['company']    ?? '' ) ),
            'event_name' => sanitize_text_field( wp_unslash( $_POST['event_name'] ?? '' ) ),
            'event_date' => sanitize_text_field( wp_unslash( $_POST['event_date'] ?? '' ) ),
            'cart'       => json_decode( stripslashes( $_POST['cart'] ?? '[]' ), true ) ?: [],
        ];

        if ( empty( $data['name'] ) || empty( $data['email'] ) || empty( $data['cart'] ) ) {
            wp_send_json_error( [ 'message' => $this->is_dutch_locale()
                ? 'Vul minimaal naam, e-mail en producten in.'
                : 'Please enter at least name, email and products.'
            ] );
            return;
        }

        $request_id = JMT_Requests::instance()->insert_request( $data );
        if ( ! $request_id ) {
            wp_send_json_error( [ 'message' => 'Fout bij het opslaan van de aanvraag.' ] );
            return;
        }

        $settings = JMT_Settings::instance();
        $to       = $settings->get_notification_email();
        $subject  = $this->is_dutch_locale()
            ? 'Nieuwe offerte-aanvraag JMT meubilair'
            : 'New JMT furniture quote request';
        $body     = $this->build_email_body( $data );
        $domain   = wp_parse_url( get_site_url(), PHP_URL_HOST );
        $headers  = [
            'Content-Type: text/html; charset=UTF-8',
            'From: JMT Browser <offerte@' . $domain . '>',
        ];

        if ( wp_mail( $to, $subject, $body, $headers ) ) {
            wp_send_json_success( [ 'message' => $this->is_dutch_locale()
                ? 'Aanvraag succesvol verzonden.'
                : 'Request sent successfully.'
            ] );
        } else {
            wp_send_json_error( [ 'message' => $this->is_dutch_locale()
                ? 'Aanvraag opgeslagen, maar e-mail kon niet worden verstuurd.'
                : 'Request saved, but the email could not be sent.'
            ] );
        }
    }

    // ── Helpers ───────────────────────────────────────────────────────────────

    private function taxon_to_label( $code, $is_nl = null ) {
        if ( $is_nl === null ) {
            $is_nl = $this->is_dutch_locale();
        }
        $map = $is_nl ? $this->category_map : $this->category_map_en;
        return $map[ $code ] ?? $map[''];
    }

    private function collect_categories( array $products, $is_nl = null ) {
        $cats = [];
        foreach ( $products as $p ) {
            $label = $this->taxon_to_label( $p['taxons']['main'] ?? '', $is_nl );
            if ( $label && ! in_array( $label, $cats, true ) ) {
                $cats[] = $label;
            }
        }
        sort( $cats );
        return $cats;
    }

    /**
     * Collect all unique primary colours across all products.
     * Uses the scalar "Primaire kleur" attribute, not array attributes.
     */
    private function collect_colors( array $products ) {
        $slugs = [];
        foreach ( $products as $p ) {
            $raw  = $this->get_primary_color( $p['attributes'] ?? [] );
            $slug = $this->color_slug( $raw );
            if ( $slug && ! in_array( $slug, $slugs, true ) ) {
                $slugs[] = $slug;
            }
        }
        sort( $slugs );
        return $slugs;
    }

    private function color_slug( $raw ) {
        if ( ! is_string( $raw ) ) return '';
        $v = trim( $raw );
        if ( $v === '' ) return '';
        $lc = strtolower( $v );

        // Normalise punctuation & whitespace
        $lc = str_replace( [ '’', '´' ], "'", $lc );
        $lc = preg_replace( '/\s+/', ' ', $lc );
        $lc = str_replace( [ ' & ', '&', ' / ', '/', ' + ' ], [ ' and ', ' and ', ' / ', ' / ', ' ' ], $lc );
        $lc = trim( $lc );

        // Known synonym map (NL + EN) → stable slug
        $map = [
            'beige' => 'beige',
            'sand' => 'sand', 'zand' => 'sand',
            'beuken' => 'beuken', 'beech' => 'beuken',
            'blauw' => 'blauw', 'blue' => 'blauw',
            'donker blauw' => 'donker-blauw', 'donkerblauw' => 'donker-blauw',
            'dark blue' => 'donker-blauw', 'darkblue' => 'donker-blauw',

            'bruin' => 'bruin-hout', 'hout' => 'bruin-hout',
            'bruin and hout' => 'bruin-hout', 'bruin & hout' => 'bruin-hout',
            'brown' => 'bruin-hout', 'wood' => 'bruin-hout',
            'brown and wood' => 'bruin-hout', 'brown & wood' => 'bruin-hout',

            'geel' => 'geel', 'yellow' => 'geel',
            'grijs' => 'grijs', 'grey' => 'grijs', 'gray' => 'grijs',

            'grey / green' => 'grey-green', 'grey/green' => 'grey-green',
            'grijs / groen' => 'grey-green', 'grijs/groen' => 'grey-green',

            'groen' => 'groen', 'green' => 'groen',
            'licht blauw' => 'licht-blauw', 'lichtblauw' => 'licht-blauw',
            'light blue' => 'licht-blauw', 'lightblue' => 'licht-blauw',

            'lungo legero' => 'lungo-legero', 'lungo legéro' => 'lungo-legero', 'lungo' => 'lungo-legero',

            'metaal / transparant' => 'metaal-transparant', 'metaal/transparant' => 'metaal-transparant',
            'metal / transparent' => 'metaal-transparant', 'metal/transparent' => 'metaal-transparant',
            'transparent' => 'metaal-transparant', 'transparant' => 'metaal-transparant',
            'metaal' => 'metaal-transparant', 'metal' => 'metaal-transparant',

            'oranje' => 'oranje', 'orange' => 'oranje',
            'paars' => 'paars', 'purple' => 'paars',

            // Special: lavendel must be light purple, not grey
            'lavendel' => 'lavendel', 'lavender' => 'lavendel',

            'rood' => 'rood', 'red' => 'rood',
            'wit' => 'wit', 'white' => 'wit',
            'zwart' => 'zwart', 'black' => 'zwart',
        ];

        if ( isset( $map[ $lc ] ) ) return $map[ $lc ];

        uksort( $map, function( $a, $b ) { return strlen( $b ) <=> strlen( $a ); } );
        foreach ( $map as $k => $slug ) {
            if ( $k !== '' && strpos( $lc, $k ) !== false ) return $slug;
        }

        $slug = preg_replace( '/[^a-z0-9]+/', '-', $lc );
        $slug = trim( $slug, '-' );
        return $slug;
    }
/**
     * Extract the primary colour from a product's attributes.
     * Looks for the "Primaire kleur" attribute specifically;
     * falls back to the first scalar attribute whose name contains "kleur"
     * (excluding frame/border colour attributes).
     */
    private function get_primary_color( array $attributes ) {
        // First pass: prefer "Primaire kleur"
        foreach ( $attributes as $attr ) {
            $name = strtolower( $attr['name'] ?? '' );
            $val  = $attr['value'] ?? null;
            if ( ! is_string( $val ) || trim( $val ) === '' || strtolower( $val ) === 'none' ) continue;
            if ( strpos( $name, 'primaire kleur' ) !== false
              || strpos( $name, 'primary color' ) !== false
              || strpos( $name, 'primary colour' ) !== false ) {
                return $val;
            }
        }
        // Second pass: any scalar "kleur" attribute that isn't frame/rand
        foreach ( $attributes as $attr ) {
            $name = strtolower( $attr['name'] ?? '' );
            $val  = $attr['value'] ?? null;
            if ( ! is_string( $val ) || trim( $val ) === '' || strtolower( $val ) === 'none' ) continue;
            $is_color = ( strpos( $name, 'kleur' ) !== false
                       || strpos( $name, 'color' ) !== false
                       || strpos( $name, 'colour' ) !== false );
            $is_excluded = ( strpos( $name, 'frame' ) !== false
                          || strpos( $name, 'rand'  ) !== false
                          || strpos( $name, 'edge'  ) !== false
                          || strpos( $name, 'border') !== false );
            if ( $is_color && ! $is_excluded ) {
                return $val;
            }
        }
        return '';
    }

    private function is_dutch_locale() {
        $locale = function_exists( 'determine_locale' ) ? determine_locale() : get_locale();
        return substr( strtolower( (string) $locale ), 0, 2 ) === 'nl';
    }

    /**
     * Build a lookup table: base_name → [ slim variant objects ]
     * A product is considered a "variant" when its name contains " - "
     * (e.g. "AMAGNI BAR SOFT - ROOD" → base "AMAGNI BAR SOFT", colour "ROOD").
     */
    private function build_variant_lookup( array $all ) {
        $lookup = [];
        foreach ( $all as $item ) {
            $name    = $item['name'] ?? '';
            $dash    = mb_strpos( $name, ' - ' );
            if ( $dash === false ) continue;

            $base    = mb_substr( $name, 0, $dash );
            $clabel  = mb_substr( $name, $dash + 3 );
            $code    = $item['code'] ?? '';
            $imgs    = array_values( array_filter(
                array_map( function( $img ) { return $img['path'] ?? ''; }, $item['images'] ?? [] )
            ) );

            $lookup[ $base ][] = [
                'id'          => $code,
                'color_label' => $clabel,
                'color_slug'  => $this->color_slug( $this->get_primary_color( $item['attributes'] ?? [] ) ?: $clabel ),
                'image'       => $imgs[0] ?? '',
            ];
        }
        return $lookup;
    }

    /**
     * Format a single product for the JSON response.
     *
     * Scalar attributes are included in `attributes` for display.
     * Array-valued attributes are never shown as attribute rows (they
     * were causing comma-joined garbage in the popup before the fix).
     *
     * `primary_color` comes from the scalar "Primaire kleur" attribute.
     * `base_name` and `color_label` are extracted from the product name
     * (e.g. "CHAIR X - ZWART" → base "CHAIR X", label "ZWART").
     * `variants` is populated later in ajax_get_products().
     */
    private function format_product( array $item, $is_nl = true ) {
        $images = array_values( array_filter(
            array_map( function( $img ) { return $img['path'] ?? ''; }, $item['images'] ?? [] )
        ) );

        $attributes = array_values( array_filter(
            $item['attributes'] ?? [],
            [ $this, 'is_displayable_attribute' ]
        ) );

        $primary_color = $this->get_primary_color( $item['attributes'] ?? [] );

        $name    = $item['name'] ?? '';
        $dash    = mb_strpos( $name, ' - ' );
        $base_name   = $dash !== false ? mb_substr( $name, 0, $dash ) : $name;
        $color_label = $dash !== false ? mb_substr( $name, $dash + 3 ) : '';

        return [
            'id'            => $item['code'] ?? '',
            'title'         => $item['name'] ?? 'Onbekend product',
            'image'         => $images[0] ?? '',
            'images_full'   => $images,
            'description'   => $item['description'] ?? '',
            'category'      => $this->taxon_to_label( $item['taxons']['main'] ?? '', $is_nl ),
            'price'         => (float) ( $item['price'] ?? 0 ),
            'attributes'    => $attributes,
            'primary_color' => $primary_color,
            'primary_color_slug' => $this->color_slug( $primary_color ),
            'base_name'     => $base_name,
            'color_label'   => $color_label,
            'variants'      => [], // populated by ajax_get_products()
        ];
    }

    /**
     * Decide whether an attribute should be shown in the product popup.
     * Arrays are excluded (the key fix from v2.1).
     */
    private function is_displayable_attribute( $attr ) {
        if ( ! is_array( $attr ) || ! array_key_exists( 'value', $attr ) ) return false;

        $v = $attr['value'];

        if ( is_array( $v ) )   return false;
        if ( is_bool( $v ) )    return $v === true;
        if ( is_null( $v ) )    return false;
        if ( is_numeric( $v ) ) return ( (float) $v ) !== 0.0;

        return trim( (string) $v ) !== '';
    }

    private function build_email_body( array $data ) {
        $is_nl = $this->is_dutch_locale();
        $rows        = '';
        $total_price = 0.0;

        foreach ( $data['cart'] as $item ) {
            $qty   = max( 1, (int) ( $item['qty']   ?? 1 ) );
            $price = (float) ( $item['price'] ?? 0 );
            $total = $qty * $price;
            $total_price += $total;
            $rows .= sprintf(
                '<tr><td>%s</td><td>%d</td><td>%s</td></tr>',
                esc_html( $item['title'] ?? '' ),
                $qty,
                number_format( $total, 2, ',', '.' )
            );
        }

        return '
        <h2>' . ( $is_nl ? 'Nieuwe offerte-aanvraag' : 'New quote request' ) . '</h2>
        <p><strong>' . ( $is_nl ? 'Naam' : 'Name' ) . ':</strong> '      . esc_html( $data['name'] )       . '</p>
        <p><strong>' . ( $is_nl ? 'E-mail' : 'Email' ) . ':</strong> '    . esc_html( $data['email'] )      . '</p>
        <p><strong>' . ( $is_nl ? 'Bedrijf' : 'Company' ) . ':</strong> '   . esc_html( $data['company'] )    . '</p>
        <p><strong>Event:</strong> '     . esc_html( $data['event_name'] ) . '</p>
        <p><strong>' . ( $is_nl ? 'Datum' : 'Date' ) . ':</strong> '     . esc_html( $data['event_date'] ) . '</p>
        <h3>' . ( $is_nl ? 'Geselecteerde producten' : 'Selected products' ) . '</h3>
        <table border="1" cellpadding="5" cellspacing="0">
          <thead><tr><th>' . ( $is_nl ? 'Product' : 'Product' ) . '</th><th>' . ( $is_nl ? 'Aantal' : 'Qty' ) . '</th><th>' . ( $is_nl ? 'Subtotaal' : 'Subtotal' ) . '</th></tr></thead>
          <tbody>' . $rows . '</tbody>
          <tfoot><tr><td colspan="2"><strong>' . ( $is_nl ? 'Totaal' : 'Total' ) . '</strong></td>
                     <td><strong>' . number_format( $total_price, 2, ',', '.' ) . '</strong></td></tr>
          </tfoot>
        </table>';
    }
}
