<?php
if ( class_exists( 'JMT_Fetcher' ) ) return;

class JMT_Fetcher {

    private static $instance;
    private $table_name;
    /** Cache directory for JSON fallback files */
    private $cache_dir;

    /** Allowed feed languages (cache partitions) */
    private $allowed_langs = [ 'nl_NL', 'en_GB' ];

    public static function instance() {
        return self::$instance ?: self::$instance = new self();
    }

    private function __construct() {
        global $wpdb;
        $this->table_name = $wpdb->prefix . 'jmt_products';
        $this->cache_dir  = JMTB_DIR . 'cache/';

        $this->maybe_install_table();
        add_action( 'init',             [ $this, 'maybe_refresh_on_init' ] );
        add_action( 'jmt_fetch_cron',   [ $this, 'do_cron_fetch' ] );
        // Non-blocking self-request handler (nopriv so it works without a session)
        add_action( 'wp_ajax_jmt_run_fetch',        [ $this, 'ajax_run_fetch' ] );
        add_action( 'wp_ajax_nopriv_jmt_run_fetch', [ $this, 'ajax_run_fetch' ] );
    }

    // ── Table management ─────────────────────────────────────────────────────

    public function maybe_install_table() {
        if ( get_option( 'jmt_products_db_version' ) !== '1.2' ) {
            $this->create_table();
            update_option( 'jmt_products_db_version', '1.2' );
        }
    }

    private function create_table() {
        global $wpdb;
        $charset = $wpdb->get_charset_collate();
        $sql = "CREATE TABLE {$this->table_name} (
            id         BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT,
            lang       VARCHAR(10)         NOT NULL DEFAULT 'nl_NL',
            data       LONGTEXT            NOT NULL,
            created_at DATETIME            NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            INDEX created_at (created_at),
            INDEX lang_created (lang, created_at)
        ) {$charset};";

        require_once ABSPATH . 'wp-admin/includes/upgrade.php';
        dbDelta( $sql );
    }

    // ── Init hook ────────────────────────────────────────────────────────────

    /**
     * On every WordPress boot: if the cache is empty or stale, trigger an
     * asynchronous background fetch.  This method never blocks the current
     * request – the page/AJAX response is sent immediately.
     */
    public function maybe_refresh_on_init() {
        // Ensure Dutch cache exists by default.
        $lang = 'nl_NL';
        if ( $this->db_has_fresh_data( $lang ) || file_exists( $this->cache_file( $lang ) ) ) return;
        $this->trigger_background_fetch( $lang );
    }

    // ── Public: get products ─────────────────────────────────────────────────

    /**
     * Returns the product array from cache.
     * If the cache is empty, triggers a background fetch and returns [].
     * The frontend's retry logic will pick up the data once the fetch completes.
     */
    public function get_products( $lang = 'nl_NL' ) {
        $lang = $this->normalize_lang( $lang );

        $products = $this->load_from_db( $lang );
        if ( $products !== false ) return $products;

        $products = $this->load_from_file( $lang );
        if ( $products !== false ) return $products;

        // No cache available. Kick off an async fetch and return empty.
        $this->trigger_background_fetch( $lang );
        return [];
    }

    // ── Background fetch ─────────────────────────────────────────────────────

    /**
     * Trigger a non-blocking asynchronous fetch via a self-targeted HTTP POST.
     * Falls back to WP-Cron if the self-request fails.
     * A transient lock prevents duplicate concurrent fetches.
     */
    public function trigger_background_fetch( $lang = 'nl_NL' ) {
        $lang = $this->normalize_lang( $lang );
        $lock = 'jmt_fetching_' . $lang;
        if ( get_transient( $lock ) ) return; // already in progress
        set_transient( $lock, time(), 5 * MINUTE_IN_SECONDS );

        // Non-blocking HTTP self-request (fires and forgets)
        $token  = wp_hash( 'jmt_fetch_' . ABSPATH . '|' . $lang );
        $result = wp_remote_post( admin_url( 'admin-ajax.php' ), [
            'timeout'   => 0.01,
            'blocking'  => false,
            'sslverify' => false,
            'body'      => [
                'action' => 'jmt_run_fetch',
                'token'  => $token,
                'lang'   => $lang,
            ],
        ] );

        // Fallback: WP-Cron if the self-request couldn't be dispatched
        if ( is_wp_error( $result ) && ! wp_next_scheduled( 'jmt_fetch_cron' ) ) {
            wp_schedule_single_event( time(), 'jmt_fetch_cron' );
        }
    }

    /** Handles the non-blocking self-request that performs the actual fetch. */
    public function ajax_run_fetch() {
        $lang = $this->normalize_lang( $_POST['lang'] ?? 'nl_NL' );
        $expected = wp_hash( 'jmt_fetch_' . ABSPATH . '|' . $lang );
        if ( ( $_POST['token'] ?? '' ) !== $expected ) {
            wp_die( '', '', [ 'response' => 403 ] );
        }

        @set_time_limit( 300 ); // large feeds may take a while
        $this->fetch_and_store( $lang );
        delete_transient( 'jmt_fetching_' . $lang );
        wp_die( '', '', [ 'response' => 200 ] );
    }

    /** WP-Cron fallback handler. */
    public function do_cron_fetch() {
        $lang = 'nl_NL';
        $this->fetch_and_store( $lang );
        delete_transient( 'jmt_fetching_' . $lang );
    }

    /** Is a background fetch currently in progress? */
    public function is_fetching() {
        foreach ( $this->allowed_langs as $lang ) {
            if ( get_transient( 'jmt_fetching_' . $lang ) ) return true;
        }
        return false;
    }

    // ── Cache info ────────────────────────────────────────────────────────────

    /**
     * Returns an associative array describing the current cache state:
     *   has_data    bool
     *   source      'database'|'bestand'|null
     *   age_seconds int|null   (seconds since last update)
     *   size_bytes  int|null   (raw byte size of stored payload)
     *   created_at  string|null (MySQL datetime)
     *   fetching    bool       (background fetch in progress)
     */
    public function get_cache_info() {
        global $wpdb;

        // Try DB
        $row = $wpdb->get_row(
            "SELECT lang, created_at, LENGTH(data) AS size_bytes
             FROM {$this->table_name}
             ORDER BY created_at DESC LIMIT 1"
        );
        if ( $row ) {
            return [
                'has_data'    => true,
                'source'      => 'database',
                'age_seconds' => time() - strtotime( $row->created_at ),
                'size_bytes'  => (int) $row->size_bytes,
                'created_at'  => $row->created_at,
                'fetching'    => $this->is_fetching(),
            ];
        }

        // Try file
        foreach ( $this->allowed_langs as $lang ) {
            $file = $this->cache_file( $lang );
            if ( file_exists( $file ) && filesize( $file ) > 0 ) {
                return [
                    'has_data'    => true,
                    'source'      => 'bestand',
                    'age_seconds' => time() - filemtime( $file ),
                    'size_bytes'  => (int) filesize( $file ),
                    'created_at'  => date( 'Y-m-d H:i:s', filemtime( $file ) ),
                    'fetching'    => $this->is_fetching(),
                ];
            }
        }

        return [
            'has_data'    => false,
            'source'      => null,
            'age_seconds' => null,
            'size_bytes'  => null,
            'created_at'  => null,
            'fetching'    => $this->is_fetching(),
        ];
    }

    // ── DB helpers ───────────────────────────────────────────────────────────

    private function db_has_fresh_data( $lang = 'nl_NL' ) {
        global $wpdb;
        $lang   = $this->normalize_lang( $lang );
        $expiry = JMT_Settings::instance()->get_cache_expiration();
        $count  = $wpdb->get_var( $wpdb->prepare(
            "SELECT COUNT(*) FROM {$this->table_name}
             WHERE lang = %s
               AND created_at >= DATE_SUB(NOW(), INTERVAL %d SECOND)",
            $lang, $expiry
        ) );
        return (int) $count > 0;
    }

    private function load_from_db( $lang = 'nl_NL' ) {
        global $wpdb;
        $lang   = $this->normalize_lang( $lang );
        $expiry = JMT_Settings::instance()->get_cache_expiration();
        $row    = $wpdb->get_row( $wpdb->prepare(
            "SELECT data FROM {$this->table_name}
             WHERE lang = %s
               AND created_at >= DATE_SUB(NOW(), INTERVAL %d SECOND)
             ORDER BY created_at DESC LIMIT 1",
            $lang, $expiry
        ) );
        if ( ! $row ) return false;

        $products = maybe_unserialize( $row->data );
        return is_array( $products ) ? $products : false;
    }

    private function store_to_db( array $products, $lang = 'nl_NL' ) {
        global $wpdb;
        $lang   = $this->normalize_lang( $lang );
        $expiry = JMT_Settings::instance()->get_cache_expiration();

        $wpdb->query( $wpdb->prepare(
            "DELETE FROM {$this->table_name} WHERE created_at < DATE_SUB(NOW(), INTERVAL %d SECOND)",
            $expiry
        ) );

        $wpdb->insert(
            $this->table_name,
            [ 'lang' => $lang, 'data' => serialize( $products ), 'created_at' => current_time( 'mysql' ) ],
            [ '%s', '%s', '%s' ]
        );
    }

    // ── File cache helpers ────────────────────────────────────────────────────

    private function load_from_file( $lang = 'nl_NL' ) {
        $file = $this->cache_file( $lang );
        if ( ! file_exists( $file ) || filesize( $file ) === 0 ) {
            return false;
        }

        $expiry = JMT_Settings::instance()->get_cache_expiration();
        if ( filemtime( $file ) < ( time() - $expiry ) ) {
            @unlink( $file );
            return false;
        }

        $json     = file_get_contents( $file );
        $products = json_decode( $json, true );
        return is_array( $products ) ? $products : false;
    }

    private function store_to_file( array $products, $lang = 'nl_NL' ) {
        $file = $this->cache_file( $lang );
        $dir  = dirname( $file );
        if ( ! file_exists( $dir ) ) {
            wp_mkdir_p( $dir );
        }

        $htaccess = $dir . '/.htaccess';
        if ( ! file_exists( $htaccess ) ) {
            file_put_contents( $htaccess, 'Deny from all' );
        }

        file_put_contents(
            $file,
            json_encode( $products, JSON_UNESCAPED_UNICODE | JSON_INVALID_UTF8_IGNORE )
        );
    }

    private function cache_file( $lang ) {
        $lang = $this->normalize_lang( $lang );
        return trailingslashit( $this->cache_dir ) . 'jmt_products_' . $lang . '.json';
    }

    private function normalize_lang( $lang ) {
        $lang = trim( (string) $lang );
        if ( $lang === '' ) return 'nl_NL';
        $lang = str_replace( '-', '_', $lang );
        $lang_lc = strtolower( $lang );

        if ( str_starts_with( $lang_lc, 'nl' ) ) return 'nl_NL';
        if ( str_starts_with( $lang_lc, 'en' ) ) return 'en_GB';

        // Fallback: if a supported lang was passed in odd casing.
        foreach ( $this->allowed_langs as $allowed ) {
            if ( strtolower( $allowed ) === $lang_lc ) return $allowed;
        }

        return 'en_GB';
    }

    // ── Remote fetch ─────────────────────────────────────────────────────────

    public function fetch_and_store( $lang = 'nl_NL' ) {
        $lang = $this->normalize_lang( $lang );
        $products = $this->fetch_from_remote( $lang );
        if ( $products === false ) return [];

        $this->store_to_db( $products, $lang );
        $this->store_to_file( $products, $lang );
        return $products;
    }

    private function fetch_from_remote( $lang = 'nl_NL' ) {
        $url = JMT_Settings::instance()->get_feed_url( $lang );

        for ( $attempt = 1; $attempt <= 3; $attempt++ ) {
            $response = wp_remote_get( $url, [ 'timeout' => 60 ] ); // 60s for large feeds

            if ( is_wp_error( $response ) ) {
                if ( $attempt < 3 ) sleep( 2 );
                continue;
            }

            if ( wp_remote_retrieve_response_code( $response ) !== 200 ) {
                if ( $attempt < 3 ) sleep( 2 );
                continue;
            }

            $body = wp_remote_retrieve_body( $response );
            $data = json_decode( $body, true );

            if ( json_last_error() !== JSON_ERROR_NONE ) continue;
            if ( empty( $data['items'] ) || ! is_array( $data['items'] ) ) continue;

            return array_map( [ $this, 'map_item' ], $data['items'] );
        }

        return false;
    }

    /**
     * Normalise a single product item from the remote feed.
     */
    private function map_item( array $item ) {
        $raw_price = $item['price']['value'] ?? 0;
        $price     = is_string( $raw_price )
            ? (float) str_replace( ',', '.', $raw_price )
            : (float) $raw_price;

        return [
            'code'        => $item['code']        ?? '',
            'name'        => $item['name']        ?? '',
            'description' => isset( $item['description'] )
                                ? substr( $item['description'], 0, 200 )
                                : '',
            'taxons'      => [
                'main' => $item['taxons']['main'] ?? '',
            ],
            'images'      => isset( $item['images'] ) && is_array( $item['images'] )
                                ? array_slice( $item['images'], 0, 3 )
                                : [],
            'price'       => $price,
            'attributes'  => isset( $item['attributes'] ) && is_array( $item['attributes'] )
                                ? $item['attributes']
                                : [],
        ];
    }
}
