/**
 * JMT Browser – Frontend JavaScript
 * Author: Dirk Zaal, Digizaal
 * Version: 2.2.8
 */
jQuery( function( $ ) {
    'use strict';

    if ( ! $( '.jmt-container' ).length ) return;

    // ── State ────────────────────────────────────────────────────────────────
    let products            = [];
    let cart                = [];
    let allCategories       = [];
    let allColors           = [];
    let allColorsAll        = [];
    let allColorsForCat     = [];
    let currentCat          = '';
    let currentColor        = '';
    let currentModalProduct = null;
    let retryCount          = 0;
    const MAX_RETRIES       = 3;
    const RETRY_DELAY       = 8000;

    /**
     * Accumulates every product ever loaded so variant swatch clicks can open
     * a full popup even when the variant lives on a different page.
     */
    const productCache = {};

    /** Cache for colour-variant families (keyed by base_name). */
    const variantFamilyCache = {};

    // ── Language selection (visitor language + easy testing) ───────────────
    function getUrlParam( key ) {
        try { return new URLSearchParams( window.location.search ).get( key ); }
        catch ( e ) { return null; }
    }

    function normaliseLang( raw ) {
        const s = String( raw || '' ).trim().toLowerCase().replace( '-', '_' );
        if ( s.indexOf( 'nl' ) === 0 ) return 'nl_NL';
        if ( s.indexOf( 'en' ) === 0 ) return 'en_GB';
        return '';
    }

    function detectVisitorLang() {
        const nav = ( navigator.languages && navigator.languages.length )
            ? navigator.languages[0]
            : ( navigator.language || '' );
        const auto = normaliseLang( nav );
        return auto || 'en_GB';
    }

    /**
     * Language rules:
     * - If ?jmt_lang=nl|en is present → explicit override AND persist (easy testing).
     * - If no query param → follow visitor language (NOT sticky).
     * - Backwards compatibility: older versions stored a plain "jmt_lang" string. We ignore that to avoid "stuck" English.
     */
    function getLang() {
        const forced = normaliseLang( getUrlParam( 'jmt_lang' ) );
        if ( forced ) {
            try { localStorage.setItem( 'jmt_lang_override', JSON.stringify( { forced: true, lang: forced } ) ); } catch ( e ) {}
            return forced;
        }

        // Remove older sticky key if present
        try { if ( localStorage.getItem( 'jmt_lang' ) ) localStorage.removeItem( 'jmt_lang' ); } catch ( e ) {}

        // Only respect persisted override if it was explicitly forced before.
        try {
            const stored = JSON.parse( localStorage.getItem( 'jmt_lang_override' ) || 'null' );
            if ( stored && stored.forced && ( stored.lang === 'nl_NL' || stored.lang === 'en_GB' ) ) {
                return stored.lang;
            }
        } catch ( e ) {}

        return detectVisitorLang();
    }

    const PREFERRED_LANG = getLang();
    const IS_NL  = PREFERRED_LANG === 'nl_NL';

    const I18N = {

        loading:              IS_NL ? 'Producten worden geladen…' : 'Loading products…',
        refreshingSupplier:   IS_NL ? '⏳ Productlijst wordt opgehaald van de leverancier…' : '⏳ Product list is being fetched from the supplier…',
        refreshingSupplier2:  IS_NL ? 'Productlijst wordt bijgewerkt van de leverancier…' : 'Product list is being updated from the supplier…',
        autoReloadIn:         IS_NL ? 'Automatisch opnieuw laden over' : 'Auto reload in',
        attempt:              IS_NL ? 'poging' : 'attempt',
        loadFailed:           IS_NL ? 'Verbinding mislukt.' : 'Connection failed.',
        serverError:          IS_NL ? 'Serverfout:' : 'Server error:',
        retry:                IS_NL ? 'Opnieuw proberen' : 'Retry',
        noImage:              IS_NL ? 'Geen afbeelding' : 'No image',
        emptySelection:       IS_NL ? 'Geen producten gevonden voor deze selectie.' : 'No products found for this selection.',
        pagesAria:            IS_NL ? 'Pagina\'s' : 'Pages',
        allColors:            IS_NL ? 'Alle kleuren' : 'All colours',
        colorVariants:        IS_NL ? 'Kleurvarianten:' : 'Colour variants:',
        addToQuote:           IS_NL ? 'Toevoegen aan offerte' : 'Add to quote',
        cartEmptyAlert:       IS_NL ? 'Uw winkelwagen is leeg. Voeg producten toe voordat u een aanvraag verstuurt.' : 'Your cart is empty. Add products before sending a request.',
        sending:              IS_NL ? 'Versturen…' : 'Sending…',
        sendRequest:          IS_NL ? 'Verstuur aanvraag' : 'Send request',
        errorLoading:         IS_NL ? 'Fout bij laden.' : 'Error while loading.',
        warmingMsg:           IS_NL ? 'Productlijst wordt opgehaald van de leverancier…' : 'Product list is being fetched from the supplier…',
    };

    // Upgrade server-rendered labels to match visitor language.
    (function syncStaticLabels(){
        if ( IS_NL ) return;
        $( '.jmt-sidebar h3' ).first().text( 'Categories' );
        $( '#jmt-color-section h3' ).text( 'Colours' );
        $( '#jmt-sidebar-quote' ).text( 'Request a quote' );
        $( '#jmt-quote-form h3' ).text( 'Request a quote' );
        $( '#jmt-products .jmt-loading' ).text( I18N.loading );

        // Quote form labels
        const labels = $( '#jmt-quote-form label' );
        if ( labels.length >= 5 ) {
            $( labels[0] ).text( 'Name *' );
            $( labels[1] ).text( 'Email *' );
            $( labels[2] ).text( 'Company' );
            $( labels[3] ).text( 'Event name' );
            $( labels[4] ).text( 'Event date' );
        }
    })();

    // ── Colour index (NL + EN synonyms → slug + label + hex) ───────────────
    // NOTE: Incoming colour strings come from the feed, so we normalise them
    // and map them to a stable slug used for CSS and UI.
    const COLOR_INDEX = [
        { slug:'beige', nl:'Beige', en:'Beige', hex:'#d8c3a5', aka:['beige'] },
        { slug:'sand', nl:'Zand', en:'Sand', hex:'#c2b280', aka:['sand','zand'] },
        { slug:'beuken', nl:'Beuken', en:'Beech', hex:'#e0c68e', aka:['beuken','beech'] },
        { slug:'blauw', nl:'Blauw', en:'Blue', hex:'#2f6db2', aka:['blauw','blue'] },
        { slug:'donker-blauw', nl:'Donker blauw', en:'Dark blue', hex:'#1a3a6b', aka:['donker blauw','donkerblauw','dark blue','darkblue'] },
        { slug:'bruin-hout', nl:'Bruin & Hout', en:'Brown & wood', hex:'#6b4f3a', aka:['bruin & hout','bruin','hout','brown & wood','brown','wood'] },
        { slug:'geel', nl:'Geel', en:'Yellow', hex:'#f2c200', aka:['geel','yellow'] },
        { slug:'grijs', nl:'Grijs', en:'Grey', hex:'#7a7a7a', aka:['grijs','grey','gray'] },
        { slug:'grey-green', nl:'Grey / Green', en:'Grey / Green', hex:'#6f7f70', aka:['grey / green','grey/green','grijs / groen','grijs/groen','grey-green'] },
        { slug:'groen', nl:'Groen', en:'Green', hex:'#2f7d4c', aka:['groen','green'] },
        { slug:'licht-blauw', nl:'Licht blauw', en:'Light blue', hex:'#74b6e6', aka:['licht blauw','lichtblauw','light blue','lightblue'] },
        { slug:'lungo-legero', nl:'Lungo Legero', en:'Lungo Legero', hex:'#a58b6a', aka:['lungo legero','lungo legéro','lungo'] },
        { slug:'metaal-transparant', nl:'Metaal / transparant', en:'Metal / transparent', hex:'#a9b0b6', aka:['metaal / transparant','metaal/transparant','metal / transparent','metal/transparent','transparent','transparant','metaal','metal'] },
        { slug:'oranje', nl:'Oranje', en:'Orange', hex:'#f28c28', aka:['oranje','orange'] },
        { slug:'paars', nl:'Paars', en:'Purple', hex:'#7b4ea3', aka:['paars','purple'] },
        { slug:'lavendel', nl:'Lavendel', en:'Lavender', hex:'#c8a2c8', aka:['lavendel','lavender'] },
        { slug:'rood', nl:'Rood', en:'Red', hex:'#d53a3a', aka:['rood','red'] },
        { slug:'wit', nl:'Wit', en:'White', hex:'#ffffff', aka:['wit','white'] },
        { slug:'zwart', nl:'Zwart', en:'Black', hex:'#111111', aka:['zwart','black'] },
    ];

    function normaliseColorKey( str ) {
        return String( str || '' )
            .trim()
            .toLowerCase()
            .replace( /\s+/g, ' ' );
    }

    function getColorMeta( raw ) {
        // If we already receive a stable slug from the server, match it directly
        for ( let i = 0; i < COLOR_INDEX.length; i++ ) {
            const row = COLOR_INDEX[i];
            if ( raw === row.slug ) return row;
        }
        const key = normaliseColorKey( raw );
        for ( let i = 0; i < COLOR_INDEX.length; i++ ) {
            const row = COLOR_INDEX[i];
            if ( row.aka.indexOf( key ) !== -1 ) return row;
        }
        // Fallback: if the feed already uses a stable name, try direct match
        for ( let i = 0; i < COLOR_INDEX.length; i++ ) {
            const row = COLOR_INDEX[i];
            if ( normaliseColorKey( row.nl ) === key || normaliseColorKey( row.en ) === key ) return row;
        }
        return { slug:'unknown', nl:raw, en:raw, hex:'#cccccc' };
    }

    function colorLabel( raw ) {
        const meta = getColorMeta( raw );
        return IS_NL ? meta.nl : meta.en;
    }

    /**
     * Returns a <span> swatch element HTML string.
     * Detects bi-color labels like "WIT / BLAUW" and renders a diagonal
     * gradient (top-left = first colour, bottom-right = second colour).
     * Falls back to a single-colour swatch class when no bi-color is found.
     */
    function renderSwatchEl( slug, rawLabel ) {
        var raw = String( rawLabel || '' ).trim();
        var slashIdx = raw.indexOf( ' / ' );
        if ( slashIdx !== -1 ) {
            // Only treat as bi-color when the full string is NOT a known single color
            // (e.g. "Grey / Green" maps to a single slug; "WIT / BLAUW" does not)
            var fullMeta = getColorMeta( raw );
            if ( fullMeta.slug === 'unknown' ) {
                var part1 = raw.substring( 0, slashIdx ).trim();
                var part2 = raw.substring( slashIdx + 3 ).trim();
                var meta1 = getColorMeta( part1 );
                var meta2 = getColorMeta( part2 );
                if ( meta1.slug !== 'unknown' && meta2.slug !== 'unknown' ) {
                    var style = 'background:linear-gradient(135deg,'
                              + meta1.hex + ' 50%,'
                              + meta2.hex + ' 50%)';
                    return '<span class="jmt-swatch" style="' + style + '"></span>';
                }
            }
        }
        var meta = getColorMeta( slug || raw );
        return '<span class="jmt-swatch jmt-swatch--' + escHtml( meta.slug ) + '"></span>';
    }

    
    function isCarpetCategoryName( name ) {
        if ( ! name ) return false;
        var n = ('' + name).toLowerCase();
        return n.indexOf('carpet') !== -1 || n.indexOf('tapijt') !== -1;
    }

    function initSidebarAccordion() {
        // Default: colours collapsed (CSS adds is-collapsed)
        $( document ).on( 'click', '.jmt-acc-toggle', function() {
            var $acc   = $( this ).closest( '.jmt-accordion' );
            var which  = $acc.data( 'acc' );
            var isOpen = ! $acc.hasClass( 'is-collapsed' );

            // toggle this one (CSS handles fast ease-in-out animation)
            if ( isOpen ) {
                $acc.addClass( 'is-collapsed' );
                $( this ).attr( 'aria-expanded', 'false' );
            } else {
                $acc.removeClass( 'is-collapsed' );
                $( this ).attr( 'aria-expanded', 'true' );

                // When opening colours, collapse categories
                if ( which === 'colors' ) {
                    var $cats = $( '.jmt-accordion--cats' );
                    $cats.addClass( 'is-collapsed' );
                    $cats.find( '.jmt-acc-toggle' ).attr( 'aria-expanded', 'false' );
                }
            }
        } );
    }

    function isAccessoriesCategoryName( name ) {
        if ( ! name ) return false;
        var n = ('' + name).toLowerCase();
        return n.indexOf( 'accessories' ) !== -1 || n.indexOf( 'accessoires' ) !== -1;
    }

    function sortCategoriesWithAccessoriesLast( cats ) {
        var list = Array.isArray( cats ) ? cats.slice() : [];
        list.sort( function( a, b ) {
            var aAcc = isAccessoriesCategoryName( a );
            var bAcc = isAccessoriesCategoryName( b );
            if ( aAcc && ! bAcc ) return 1;
            if ( bAcc && ! aAcc ) return -1;
            return (''+a).localeCompare( ''+b, undefined, { sensitivity: 'base' } );
        } );
        return list;
    }

    function getVariantFamily( p ) {
        var key = p && p.base_name ? p.base_name : '';
        if ( ! key ) return [];

        if ( variantFamilyCache[ key ] ) return variantFamilyCache[ key ];

        var family = [];

        // Current product
        family.push( {
            id: p.id,
            slug: p.primary_color_slug || getColorMeta( p.color_label ).slug,
            label: colorLabel( p.primary_color_slug || p.color_label ),
            rawLabel: p.color_label || p.primary_color || '',
            image: p.image || ''
        } );

        // Siblings
        if ( Array.isArray( p.variants ) ) {
            p.variants.forEach( function( v ) {
                var slug = v.color_slug || getColorMeta( v.color_label ).slug;
                family.push( {
                    id: v.id,
                    slug: slug,
                    label: colorLabel( v.color_slug || v.color_label ),
                    rawLabel: v.color_label || '',
                    image: v.image || ''
                } );
            } );
        }

        // Stable order: by slug then by id
        family.sort( function( a, b ) {
            if ( a.slug < b.slug ) return -1;
            if ( a.slug > b.slug ) return 1;
            return (''+a.id).localeCompare(''+b.id);
        } );

        variantFamilyCache[ key ] = family;
        return family;
    }
// ── Hamburger / sidebar toggle (mobile) ──────────────────────────────────
    function closeMobileSidebar() {
        $( '.jmt-sidebar' ).removeClass( 'is-open' );
        $( '.jmt-menu-toggle' ).attr( 'aria-expanded', 'false' );
        $( '.jmt-menu-toggle-icon' ).text( '\u2630' );
    }

    $( document ).on( 'click', '.jmt-menu-toggle', function() {
        var $sidebar = $( '.jmt-sidebar' );
        var opening  = ! $sidebar.hasClass( 'is-open' );
        $sidebar.toggleClass( 'is-open' );
        $( this ).attr( 'aria-expanded', opening ? 'true' : 'false' );
        $( this ).find( '.jmt-menu-toggle-icon' ).text( opening ? '\u2715' : '\u2630' );
    } );

// ── Boot ─────────────────────────────────────────────────────────────────
    initSidebarAccordion();
    loadProducts();
    updateCartUI();

    // ── Load products from server ─────────────────────────────────────────────
    function loadProducts( page, category, color, isRetry ) {
        if ( page     === undefined ) page     = 1;
        if ( category === undefined ) category = currentCat;
        if ( color    === undefined ) color    = currentColor;
        if ( isRetry  === undefined ) isRetry  = false;

        if ( ! isRetry ) retryCount = 0;

        var loadingMsg = retryCount > 0
            ? '<p class="jmt-loading jmt-refreshing">' + I18N.refreshingSupplier2
              + '<br><small>' + I18N.autoReloadIn + ' ' + ( RETRY_DELAY / 1000 ) + 's'
              + ' (' + I18N.attempt + ' ' + retryCount + ' / ' + MAX_RETRIES + ')</small></p>'
            : '<p class="jmt-loading">' + I18N.loading + '</p>';

        $( '#jmt-products' ).html( loadingMsg );

        $.post( JMT_Ajax.ajax_url, {
            action:   'jmt_get_products',
            nonce:    JMT_Ajax.nonce,
            lang:     PREFERRED_LANG,
            page:     page,
            category: category,
            color:    color
        } )
        .done( function( res ) {
            if ( ! res.success ) {
                var isWarming = res.data && res.data.status === 'warming';
                if ( isWarming && retryCount < MAX_RETRIES ) {
                    retryCount++;
                    $( '#jmt-products' ).html(
                        '<p class="jmt-loading jmt-refreshing">'
                        + I18N.refreshingSupplier
                        + '<br><small>' + I18N.autoReloadIn + ' ' + ( RETRY_DELAY / 1000 ) + 's'
                        + ' (' + I18N.attempt + ' ' + retryCount + ' / ' + MAX_RETRIES + ')</small></p>'
                    );
                    setTimeout( function() {
                        loadProducts( page, category, color, true );
                    }, RETRY_DELAY );
                } else {
                    var msg = res.data && res.data.message ? res.data.message : I18N.errorLoading;
                    $( '#jmt-products' ).html(
                        '<p class="jmt-error">' + escHtml( msg )
                        + ' <button class="jmt-retry-btn jmt-button">' + I18N.retry + '</button></p>'
                    );
                }
                return;
            }

            retryCount    = 0;
            products      = res.data.products   || [];
            allCategories = sortCategoriesWithAccessoriesLast( res.data.categories || [] );

            allColorsAll    = res.data.colors || [];
            allColorsForCat = res.data.colors_in_category || allColorsAll;
            allColors       = currentCat ? allColorsForCat : allColorsAll;

            // If the current colour isn't available for this category, reset it.
            if ( currentColor && allColors.indexOf( currentColor ) === -1 ) {
                currentColor = '';
            }

            // Accumulate all loaded products into the cache
            products.forEach( function( p ) { productCache[ p.id ] = p; } );

            renderCategories();
            renderColors();
            renderProducts( products );
            renderPagination( res.data.page, Math.ceil( res.data.total / res.data.per_page ) );

            // Auto-select first category on initial load
            if ( ! currentCat && allCategories.length ) {
                currentCat = allCategories[0];
                $( '.jmt-category[data-cat="' + currentCat + '"]' ).addClass( 'selected' );
                loadProducts( 1, currentCat, '' );
            }
        } )
        .fail( function( xhr, status, error ) {
            if ( retryCount < MAX_RETRIES ) {
                retryCount++;
                $( '#jmt-products' ).html(
                    '<p class="jmt-loading jmt-refreshing">'
                    + I18N.loadFailed + ' ' + I18N.autoReloadIn + ' ' + ( RETRY_DELAY / 1000 ) + 's…'
                    + '<br><small>(' + I18N.attempt + ' ' + retryCount + ' / ' + MAX_RETRIES + ')</small></p>'
                );
                setTimeout( function() {
                    loadProducts( page, category, color, true );
                }, RETRY_DELAY );
            } else {
                $( '#jmt-products' ).html(
                    '<p class="jmt-error">' + I18N.serverError + ' ' + escHtml( error )
                    + ' <button class="jmt-retry-btn jmt-button">' + I18N.retry + '</button></p>'
                );
            }
        } );
    }

    // Manual retry button
    $( '#jmt-products' ).on( 'click', '.jmt-retry-btn', function() {
        retryCount = 0;
        loadProducts( 1, currentCat, currentColor );
    } );

    // ── Render helpers ────────────────────────────────────────────────────────

    function renderCategories() {
        var $list = $( '#jmt-category-filter' ).empty();
        allCategories.forEach( function( cat ) {
            var cls = cat === currentCat ? ' selected' : '';
            $list.append( '<li class="jmt-category' + cls + '" data-cat="' + escHtml( cat ) + '">' + escHtml( cat ) + '</li>' );
        } );
    }

    function renderColors() {
        var $section = $( '#jmt-color-section' );
        var $list    = $( '#jmt-color-filter' ).empty();

        if ( ! allColors.length ) {
            $section.hide();
            return;
        }

        $section.show();

        var allCls = currentColor === '' ? ' selected' : '';
        $list.append(
            '<li class="jmt-color' + allCls + '" data-color="">'
            + '<span class="jmt-color-item">'
            + '<span class="jmt-swatch jmt-swatch--all"></span>'
            + '<span class="jmt-color-name">' + escHtml( I18N.allColors ) + '</span>'
            + '</span>'
            + '</li>'
        );

        allColors.forEach( function( color ) {
            var cls   = color === currentColor ? ' selected' : '';
            var meta  = getColorMeta( color );
            var label = colorLabel( color );
            $list.append(
                '<li class="jmt-color' + cls + '" data-color="' + escHtml( color ) + '">' 
                + '<span class="jmt-color-item">'
                + '<span class="jmt-swatch jmt-swatch--' + escHtml( meta.slug ) + '"></span>'
                + '<span class="jmt-color-name">' + escHtml( label ) + '</span>'
                + '</span>'
                + '</li>'
            );
        } );
    }

    function renderProducts( list ) {
        var $grid = $( '#jmt-products' ).empty();
        $( '#jmt-modal' ).hide();

        if ( ! list.length ) {
            $grid.html( '<p class="jmt-empty">' + I18N.emptySelection + '</p>' );
            return;
        }

        list.forEach( function( p ) {
            var img = p.image
                ? '<img src="' + escHtml( p.image ) + '" alt="' + escHtml( p.title ) + '" loading="lazy">'
                : '<div class="jmt-no-image">' + I18N.noImage + '</div>';
            $grid.append(
                '<div class="jmt-product" data-id="' + escHtml( p.id ) + '">'
                + img
                + '<h4>' + escHtml( p.title ) + '</h4>'
                + '</div>'
            );
        } );
    }

    function renderPagination( currentPage, totalPages ) {
        $( '.jmt-pagination' ).remove();
        if ( totalPages <= 1 ) return;

        var $nav  = $( '<nav class="jmt-pagination" aria-label="' + I18N.pagesAria + '"></nav>' );
        var pages = [];

        pages.push( pageBtn( '&laquo;', 1 ) );
        if ( currentPage > 3 )             pages.push( pageBtn( '1', 1 ) );
        if ( currentPage > 4 )             pages.push( '<span class="jmt-ellipsis">&hellip;</span>' );
        if ( currentPage > 1 )             pages.push( pageBtn( currentPage - 1, currentPage - 1 ) );
        pages.push( pageBtn( currentPage, currentPage, true ) );
        if ( currentPage < totalPages )     pages.push( pageBtn( currentPage + 1, currentPage + 1 ) );
        if ( currentPage < totalPages - 3 ) pages.push( '<span class="jmt-ellipsis">&hellip;</span>' );
        if ( currentPage < totalPages - 2 ) pages.push( pageBtn( totalPages, totalPages ) );
        pages.push( pageBtn( '&raquo;', totalPages ) );

        $nav.html( pages.join( '' ) );
        $( '#jmt-products' ).after( $nav );
    }

    function pageBtn( label, page, active ) {
        var cls = 'jmt-page' + ( active ? ' active' : '' );
        return '<button class="' + cls + '" data-page="' + page + '">' + label + '</button>';
    }

    // ── Product detail modal ──────────────────────────────────────────────────

    $( '#jmt-products' ).on( 'click', '.jmt-product', function() {
        var id = $( this ).data( 'id' );
        var p  = products.find( function( x ) { return x.id === id; } );
        if ( ! p ) return;
        openProductModal( p );
    } );

    function openProductModal( p ) {
        currentModalProduct = p;

        // ── Left column: image, title, colour variant swatches, description ──
        var leftHtml = p.image
            ? '<img id="jmt-modal-main-img" src="' + escHtml( p.image ) + '" alt="' + escHtml( p.title ) + '" loading="lazy">'
            : '<div class="jmt-no-image">' + I18N.noImage + '</div>';

        leftHtml += '<h2>' + escHtml( p.title ) + '</h2>';

        // Show colour-variant swatches when the product belongs to a named
        // variant family (name contains " - ") or has sibling variants.
        var showVariants = ! isCarpetCategoryName( currentCat ) && ! isCarpetCategoryName( p.category );

        var family = showVariants ? getVariantFamily( p ) : [];
        if ( showVariants && family.length > 1 ) {
            leftHtml += '<div class="jmt-modal-colors"><strong>' + I18N.colorVariants + '</strong>'
                      + '<div class="jmt-color-swatches">';

            family.forEach( function( v ) {
                var activeCls = ( currentModalProduct && v.id === currentModalProduct.id ) ? ' active' : '';
                leftHtml += '<button class="jmt-color-swatch' + activeCls + '"'
                          + ' data-variant-id="' + escHtml( v.id ) + '"'
                          + ' data-variant-img="' + escHtml( v.image || '' ) + '"'
                          + ' title="' + escHtml( v.label ) + '"'
                          + ' aria-label="' + escHtml( v.label ) + '">'
                          + renderSwatchEl( v.slug, v.rawLabel )
                          + '</button>';
            } );

            leftHtml += '</div></div>';
        }


        if ( p.description ) {
            leftHtml += '<p class="jmt-description">' + escHtml( p.description ) + '</p>';
        }

        // ── Right column: attributes grid + cart controls ─────────────────────
        var attrsHtml = '<div class="jmt-attr-grid">';

        if ( Array.isArray( p.attributes ) ) {
            p.attributes.forEach( function( attr ) {
                var v = attr.value;

                if ( v === null || v === undefined )            return;
                if ( Array.isArray( v ) )                       return;
                if ( v === false || v === 0 || v === '' )       return;
                if ( typeof v === 'string' && v.trim() === '' ) return;

                var display = v;
                if ( v === true ) {
                    display = '✅ ' + attr.name;
                } else if ( typeof v === 'number' ) {
                    var cmCodes = [ 'depth', 'width', 'height', 'seat_height', 'total_height', 'diameter' ];
                    if ( cmCodes.indexOf( attr.code ) !== -1 ) {
                        display = v + ' cm';
                    }
                }

                attrsHtml += '<div class="attr-name">' + escHtml( attr.name ) + '</div>'
                           + '<div class="attr-value">' + escHtml( String( display ) ) + '</div>';
            } );
        }

        attrsHtml += '</div>';

        attrsHtml += '<div class="jmt-cart-controls">'
                   + '<input type="number" id="jmt-qty-' + escHtml( p.id ) + '" value="1" min="1" class="jmt-cart-qty">'
                   + '<button class="jmt-button jmt-add-cart" data-id="' + escHtml( p.id ) + '">' + I18N.addToQuote + '</button>'
                   + '</div>';

        var body = '<div class="jmt-modal-cols">'
                 + '<div class="jmt-modal-left">' + leftHtml + '</div>'
                 + '<div class="jmt-modal-right">' + attrsHtml + '</div>'
                 + '</div>';

        $( '#jmt-modal-body' ).html( body );
        $( '#jmt-modal' ).show();
    }

    // ── Colour variant swatch click inside modal ──────────────────────────────

    $( '#jmt-modal-body' ).on( 'click', '.jmt-color-swatch', function() {
        var variantId  = $( this ).data( 'variant-id' );
        var variantImg = $( this ).data( 'variant-img' );

        // Clicking the currently-shown product's swatch: do nothing
        if ( currentModalProduct && variantId === currentModalProduct.id ) return;

        $( '.jmt-color-swatch' ).removeClass( 'active' );
        $( this ).addClass( 'active' );

        // If the full product is in cache, open its complete popup
        if ( productCache[ variantId ] ) {
            openProductModal( productCache[ variantId ] );
            return;
        }

        // Fallback: just swap the image (no full data available yet)
        if ( variantImg ) {
            $( '#jmt-modal-main-img' ).attr( 'src', variantImg );
        }
    } );

    // Close modal
    $( '#jmt-modal' ).on( 'click', function( e ) {
        if ( e.target.id === 'jmt-modal' || e.target.id === 'jmt-modal-close' ) {
            $( '#jmt-modal' ).hide();
        }
    } );

    // ── Cart ──────────────────────────────────────────────────────────────────

    $( '#jmt-modal' ).on( 'click', '.jmt-add-cart', function() {
        var id  = $( this ).data( 'id' );
        var p   = productCache[ id ] || products.find( function( x ) { return x.id === id; } );
        if ( ! p ) return;

        var qty      = Math.max( 1, parseInt( $( '#jmt-qty-' + id ).val() ) || 1 );
        var existing = cart.find( function( x ) { return x.id === id; } );
        if ( existing ) {
            existing.qty += qty;
        } else {
            cart.push( $.extend( {}, p, { qty: qty } ) );
        }

        updateCartUI();
        $( '#jmt-modal' ).hide();
    } );

    // Remove item from cart
    $( '#jmt-cart' ).on( 'click', '.jmt-cart-remove', function() {
        var id = $( this ).data( 'id' );
        cart   = cart.filter( function( x ) { return x.id !== id; } );
        updateCartUI();
    } );

    function updateCartUI() {
        var $list  = $( '#jmt-cart' ).empty();
        var $empty = $( '#jmt-cart-empty' );
        var $btn   = $( '#jmt-submit-quote' );

        if ( ! cart.length ) {
            $empty.show();
            $btn.prop( 'disabled', true );
            return;
        }

        $empty.hide();
        $btn.prop( 'disabled', false );

        cart.forEach( function( item ) {
            $list.append(
                '<li class="jmt-cart-item">'
                + '<span class="jmt-cart-title">' + escHtml( item.title ) + '</span>'
                + ' <span class="jmt-cart-qty-badge">x' + item.qty + '</span>'
                + ' <button class="jmt-cart-remove" data-id="' + escHtml( item.id ) + '" title="' + ( IS_NL ? 'Verwijderen' : 'Remove' ) + '">&times;</button>'
                + '</li>'
            );
        } );
    }

    // Scroll to quote form
    $( '#jmt-sidebar-quote' ).on( 'click', function() {
        var offset = $( '#jmt-quote-form' ).offset();
        if ( offset ) {
            $( 'html, body' ).animate( { scrollTop: offset.top - 80 }, 500 );
        }
    } );

    // ── Category filter ───────────────────────────────────────────────────────

    $( '#jmt-category-filter' ).on( 'click', 'li.jmt-category', function() {
        var cat = $( this ).data( 'cat' );
        $( '.jmt-category' ).removeClass( 'selected' );
        $( this ).addClass( 'selected' );
        currentCat   = cat;
        currentColor = '';  // reset colour filter when switching category
        $( '.jmt-pagination' ).remove();
        if ( $( window ).width() <= 640 ) { closeMobileSidebar(); }
        loadProducts( 1, currentCat, '' );
        var topOffset = $( '#jmt-products' ).offset();
        if ( topOffset ) {
            $( 'html, body' ).animate( { scrollTop: topOffset.top - 100 }, 300 );
        }
    } );

    // ── Colour filter ──────────────────────────────────────────────────────────

    $( '#jmt-color-filter' ).on( 'click', 'li.jmt-color', function() {
        var color = $( this ).data( 'color' );
        $( '.jmt-color' ).removeClass( 'selected' );
        $( this ).addClass( 'selected' );
        currentColor = color;
        $( '.jmt-pagination' ).remove();
        if ( $( window ).width() <= 640 ) { closeMobileSidebar(); }
        loadProducts( 1, currentCat, currentColor );
    } );

    // ── Pagination ────────────────────────────────────────────────────────────

    $( document ).on( 'click', '.jmt-page', function() {
        if ( $( this ).hasClass( 'active' ) ) return;
        var topOffset = $( '#jmt-products' ).offset();
        if ( topOffset ) {
            $( 'html, body' ).animate( { scrollTop: topOffset.top - 80 }, 300 );
        }
        loadProducts( parseInt( $( this ).data( 'page' ) ), currentCat, currentColor );
    } );

    // ── Quote form submit ─────────────────────────────────────────────────────

    $( '#jmt-quote-form' ).on( 'submit', function( e ) {
        e.preventDefault();

        if ( ! cart.length ) {
            alert( I18N.cartEmptyAlert );
            return;
        }

        var $form = $( this );
        var $btn  = $form.find( '#jmt-submit-quote' );
        $btn.prop( 'disabled', true ).text( I18N.sending );

        $form.find( '#jmt-cart-input' ).val( JSON.stringify( cart ) );

        $.post( JMT_Ajax.ajax_url, {
            action:     'jmt_send_quote',
            nonce:      JMT_Ajax.nonce,
            name:       $form.find( '[name="name"]'       ).val(),
            email:      $form.find( '[name="email"]'      ).val(),
            company:    $form.find( '[name="company"]'    ).val(),
            event_name: $form.find( '[name="event_name"]' ).val(),
            event_date: $form.find( '[name="event_date"]' ).val(),
            cart:       JSON.stringify( cart )
        } )
        .done( function( res ) {
            if ( res.success ) {
                alert( res.data.message );
                $form[0].reset();
                cart = [];
                updateCartUI();
            } else {
                alert( res.data && res.data.message ? res.data.message : 'Er is een fout opgetreden.' );
            }
        } )
        .fail( function( xhr, status, error ) {
            alert( ( IS_NL ? 'Versturen mislukt: ' : 'Send failed: ' ) + error );
        } )
        .always( function() {
            $btn.prop( 'disabled', false ).text( I18N.sendRequest );
        } );
    } );

    // ── Utility ───────────────────────────────────────────────────────────────

    function escHtml( str ) {
        return String( str )
            .replace( /&/g,  '&amp;'  )
            .replace( /</g,  '&lt;'   )
            .replace( />/g,  '&gt;'   )
            .replace( /"/g,  '&quot;' )
            .replace( /'/g,  '&#039;' );
    }

} );
